function ball_motion_simulation()
    % Parameter settings
    numBalls = 100;
    diskDiameter = 90; % mm
    ballDiameters = [6; 3*ones(numBalls-1, 1)]; % Diameter of the first ball is 6mm, others are 3mm
    ballProbabilities = [0.3; 0.6*ones(numBalls-1, 1)]; % The first ball has a movement probability of 0.3, others 0.6
    colors = ['r'; repmat('b', numBalls-1, 1)]; % The first ball is red, others are blue
    q1 = 3; % Power-law exponent for the first ball's direction update
    q2 = 2; % Power-law exponent for the other balls' direction update
    numSteps = 2000; % Number of simulation steps

    % Initialize positions and directions of the balls
    positions = initialize_positions(numBalls, diskDiameter, ballDiameters);
    directions = rand(numBalls, 1) * 2 * pi; % Random initial directions
    lastUpdateTimes = zeros(numBalls, 1); % Record the last time the direction was updated
    
    % Trajectories and encounter counts
    trajectories = cell(numBalls, 1);
    for i = 1:numBalls
        trajectories{i} = zeros(numSteps, 2);
    end
    encounterCounts = zeros(numBalls, numBalls);
    
    % Create animation window
    figure;
    axis equal;
    hold on;
    viscircles([0, 0], diskDiameter / 2, 'LineStyle', '-','Color','b'); % Draw disk boundary
    ballPlots = gobjects(numBalls, 1);
    for i = 1:numBalls
        ballPlots(i) = plot(positions(i, 1), positions(i, 2), 'o', 'MarkerFaceColor', colors(i), 'MarkerEdgeColor', colors(i), 'MarkerSize', ballDiameters(i));
    end
    hold off;
    xlim([-diskDiameter / 2, diskDiameter / 2]);
    ylim([-diskDiameter / 2, diskDiameter / 2]);
    title('Individual Motion Simulation');
    
    % Create video writer object
    v = VideoWriter('ball_motion_simulation.avi');
    open(v);

    % Simulation of the motion process
    for t = 1:numSteps
        for i = 1:numBalls
            if rand < ballProbabilities(i)
                % Move one step
                stepSize = 1; % Step length, can be adjusted as needed
                newPos = positions(i, :) + stepSize * [cos(directions(i)), sin(directions(i))];
                if is_inside_disk(newPos, diskDiameter)
                    positions(i, :) = newPos;
                else
                    % If the ball hits the disk boundary, bounce back in a random direction
                    directions(i) = rand * 2 * pi;
                end
            end
            
            % Update animation
            set(ballPlots(i), 'XData', positions(i, 1), 'YData', positions(i, 2));
            
            % Record trajectory
            trajectories{i}(t, :) = positions(i, :);
        end
        
        % Check for encounters and update directions
        for i = 1:numBalls
            for j = i+1:numBalls
                if pdist2(positions(i, :), positions(j, :)) < (ballDiameters(i) + ballDiameters(j)) / 2
                    encounterCounts(i, j) = encounterCounts(i, j) + 1;
                    encounterCounts(j, i) = encounterCounts(j, i) + 1;
                    
                    % Check direction update conditions
                    if should_update_direction(t - lastUpdateTimes(i), q1)
                        directions(i) = rand * 2 * pi;
                        lastUpdateTimes(i) = t;
                    end
                    if should_update_direction(t - lastUpdateTimes(j), q2)
                        directions(j) = rand * 2 * pi;
                        lastUpdateTimes(j) = t;
                    end
                end
            end
        end
        
        drawnow;
        % Write the current frame to video
        frame = getframe(gcf);
        writeVideo(v, frame);
    end
    
    % Close video writer object
    close(v);

    % Save trajectories and encounter counts
    save('trajectories.mat', 'trajectories');
    save('encounterCounts.mat', 'encounterCounts');
    
    % Plot the trajectories of the first and second balls
    figure;
    hold on;
    viscircles([0, 0], diskDiameter / 2, 'LineStyle', '-','Color','b'); % Draw disk boundary
    plot(trajectories{1}(:, 1), trajectories{1}(:, 2), 'r');
    plot(trajectories{2}(:, 1), trajectories{2}(:, 2), 'b');
    xlabel('X');
    ylabel('Y');
    title('Trajectories of the First and Second Balls');
    legend('First Ball', 'Second Ball');
    xlim([-diskDiameter / 2, diskDiameter / 2]);
    ylim([-diskDiameter / 2, diskDiameter / 2]);
    axis equal;
    hold off;
end

function positions = initialize_positions(numBalls, diskDiameter, ballDiameters)
    positions = zeros(numBalls, 2);
    for i = 1:numBalls
        while true
            pos = (rand(1, 2) - 0.5) * diskDiameter;
            if is_inside_disk(pos, diskDiameter) && all(pdist2(pos, positions(1:i-1, :)) >= ballDiameters(i))
                positions(i, :) = pos;
                break;
            end
        end
    end
end

function inside = is_inside_disk(pos, diskDiameter)
    inside = norm(pos) <= diskDiameter / 2;
end

function update = should_update_direction(interval, q)
    update = interval >= powerlaw_interval(q);
end

function interval = powerlaw_interval(q)
    interval = rand^(1/(q-1));
end
